#from analogDiscover import *
import muriTileV2
import time

class muriArrayV2:
    # coordinateList is a list of (x,y) tuples
    # address list is the order that data is programmed
    def __init__(self, spi_channel, addressList, coordinateList, broadcast_address,state_mapper = None):
        assert(len(addressList) == len(coordinateList))
        print "Creating array..."
        self.addressList = addressList
        self.numTiles = len(addressList)
        self.spi_channel = spi_channel
        self.broadcast_address = broadcast_address
        self.state_mapper = state_mapper

        self.tiles = {}
        print "Creating tiles..."
        for i in range(len(addressList)):
            coordinate = coordinateList[i]
            address = addressList[i]
            self.tiles[address] = muriTileV2.muriTileV2(address,coordinate,self.spi_channel)
        print "Flushing channel..."
        # for some reason, array initially behaves like a buffer for the first 50 messages
        for i in range(54):
            self.arrayEnableRF() # send 53 useless messages
        print "Array ready for programming"

    def getTileByCoordinates(self,coordinates):
        print "ASKING FOR TILE COORDS"
        for addr in self.tiles.keys():
            tile = self.tiles[addr]
            if tile.coordinates == coordinates:
                return tile
        return self.tiles[coordinates]

    def getTileByAddress(self,address):
        return self.tiles[address]

# ========================= ARRAY PROGRAMMING METHODS ==============================

    # turn off all RF power in the array
    def arrayEnableRF(self,verbose=False):
        metadata = 1
        command = 32
        data = [metadata, self.broadcast_address,command]
        self.spi_channel.sendData(data,verbose)

    # turn on all RF power in the array
    def arrayDisableRF(self,verbose=False):
        metadata = 1
        command = 16
        data = [metadata, self.broadcast_address,command]
        self.spi_channel.sendData(data,verbose)

    # programs output powers of each tile so equal power is measured at theta = 0
    # power settings are hardcoded based on analysis performed on 11/16/21 with ground plane
    def normalizeTileOutputPower(self):
        normed_pwr_settings = {13:11,1:11,2:10,3:13,4:11,5:12,6:11,7:12,8:10,9:12,10:15,11:13,12:10,14:11,15:12,16:11,17:10,18:11,19:11,20:13,21:11,22:10,23:11,24:10,25:12}
        pwr_settings = []
        for addr in self.addressList:
            pwr_settings += [normed_pwr_settings[addr]]
        self.programArrayOutputPower(pwr_settings)

    # program the output power of every tile in the array
    # phase data should be added in order of address list
    # use broadcast option to send same data to every element in array
    def programArrayOutputPower(self,power_data,broadcast = False, verbose=False):
        if broadcast:
            metadata = 65
            command = 2
            data = [metadata,self.broadcast_address,command, power_data]
        else:
            assert(len(power_data) == self.numTiles)
            metadata = 64 + self.numTiles
            command = 2
            data = [metadata]
            for tile_idx in range(self.numTiles):
                tile_packet = [ self.addressList[tile_idx], command]
                power = power_data[tile_idx]
                tile_packet += [power]
                data += tile_packet
        self.spi_channel.sendData(data,verbose)

    # program the phases of every tile in the array
    # phase data should be added in order of address list
    def programArrayPhase(self,phase_data,verbose=False):
        assert(len(phase_data) == self.numTiles)
        metadata = 128 + self.numTiles
        command = 4
        data = [metadata]
        for tile_idx in range(self.numTiles):
            tile_packet = [ self.addressList[tile_idx], command]
            phase = phase_data[tile_idx]
            tile_packet += muriArrayV2.convertPhaseToBytes(phase)
            data += tile_packet
        self.spi_channel.sendData(data,verbose)

    # enable on all meta gaps in the array
    def arrayEnableMetagaps(self,verbose=False):
        metadata = 1
        command = 128
        data = [metadata, self.broadcast_address,command]
        self.spi_channel.sendData(data,verbose)

    # turn off all meta gaps in the array
    def arrayDisableMetagaps(self,verbose=False):
        metadata = 1
        command = 64
        data = [metadata, self.broadcast_address,command]
        self.spi_channel.sendData(data,verbose)

    # turn on all meta gaps in the array
    def arrayTurnOnAllMetagaps(self,verbose=False):
        metadata = 33
        command = 8
        data = [metadata, self.broadcast_address,command]
        data += [255,255,255,255,255,255]
        self.spi_channel.sendData(data,verbose)

    # program the meta gap state of every tile in the array
    # data should be added in order of address list
    # use broadcast option to send same data to every element in array
    def programArrayMetaGaps(self,mg_data_e,mg_data_h=None,broadcast = False, verbose=False):
        if broadcast:
            metadata = 33
            command = 8
            data =[metadata,self.broadcast_address,command]
            data += muriArrayV2.convertMetagapStateToBytes(mg_data_e)
            if mg_data_h is not None:
                data += muriArrayV2.convertMetagapStateToBytes(mg_data_h)
            else:
                data += muriArrayV2.convertMetagapStateToBytes(mg_data_e)
        else:
            assert(len(mg_data_e) == self.numTiles)
            metadata = 32 + self.numTiles
            command = 8
            data = [metadata]
            for tile_idx in range(self.numTiles):
                tile_packet = [self.addressList[tile_idx], command]
                mg_state_e = mg_data_e[tile_idx]
                tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_e)
                if mg_data_h is not None:
                    mg_state_h = mg_data_h[tile_idx]
                    tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_h)
                else:
                    tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_e)
                data += tile_packet
        self.spi_channel.sendData(data,verbose)
        if not broadcast: # sending multiple large commands in a row can result in some of the elements "freezing"
            time.sleep(0.005)

    # command is command data byte
    # phase data is in degrees
    # mg data is a state number
    def directProgramArray(self,commands,phase_data = None,power_data = None,mg_data_e = None, mg_data_h = None,verbose=False):
        assert(len(commands) == self.numTiles)
        program_phase = (phase_data is not None)
        program_power = (power_data is not None)
        program_mg = (mg_data_e is not None)

        metadata = muriArrayV2.getPacketMetadata(self.numTiles,program_phase,program_power,program_mg)
        data = [metadata]
        for tile_idx in range(self.numTiles):
            tile_packet = [ self.addressList[tile_idx], commands[tile_idx]]
            if program_phase:
                phase = phase_data[tile_idx]
                tile_packet += muriArrayV2.convertPhaseToBytes(phase)
            if program_mg:
                mg_state_e = mg_data_e[tile_idx]
                mg_state_h = mg_data_h[tile_idx]
                tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_e)
                if mg_data_h is not None:
                    tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_h)
                else:
                    tile_packet += muriArrayV2.convertMetagapStateToBytes(mg_state_e)
            if program_power:
                power = power_data[tile_idx]
                tile_packet += [power]
            data += tile_packet
        self.spi_channel.sendData(data,verbose)

    # iterates through every switch and uses mapper to determine which state bit determines its status
    # then converts this information into states for each sheet to be programmed with
    def convertStateToSheetData(self,state):
        sheet_states_e = []
        sheet_states_h = []
        for tile_idx in range(self.numTiles):
            tile = self.getTileByAddress(self.addressList[tile_idx])
            tile_coordinates = tile.coordinates
            # calculate state for E plane Sheet
            sheet_state_e = 0
            for x_row in range(4):
                for x_col in range(3):
                    # Use mapper to get what bit in the state controls this switch setting
                    bit_idx = self.state_mapper.getStateBitIdx(tile_coordinates,'e','x',x_row,x_col)
                    # Identify value of that bit
                    sw_state = (state >> bit_idx) & 1
                    # Update sheet state to program switch to correct setting
                    if sw_state:
                        bit_loc = 23 - (x_col + 3*x_row)
                        sheet_state_e += 2**bit_loc
            for y_row in range(3):
                for y_col in range(4):
                    # Use mapper to get what bit in the state controls this switch setting
                    bit_idx = self.state_mapper.getStateBitIdx(tile_coordinates,'e','y',y_row,y_col)
                    # Identify value of that bit
                    sw_state = (state >> bit_idx) & 1
                    # Update sheet state to program switch to correct setting
                    if sw_state:
                        bit_loc = 11 - (y_col + 4*y_row)
                        sheet_state_e += 2**bit_loc
            sheet_states_e += [sheet_state_e]

            # calculate state for H plane Sheet
            sheet_state_h = 0
            for x_row in range(4):
                for x_col in range(3):
                    # Use mapper to get what bit in the state controls this switch setting
                    bit_idx = self.state_mapper.getStateBitIdx(tile_coordinates,'h','x',x_row,x_col)
                    # Identify value of that bit
                    sw_state = (state >> bit_idx) & 1
                    # Update sheet state to program switch to correct setting
                    if sw_state:
                        bit_loc = 23 - (x_col + 3*x_row)
                        sheet_state_h += 2**bit_loc
            for y_row in range(3):
                for y_col in range(4):
                    # Use mapper to get what bit in the state controls this switch setting
                    bit_idx = self.state_mapper.getStateBitIdx(tile_coordinates,'h','y',y_row,y_col)
                    # Identify value of that bit
                    sw_state = (state >> bit_idx) & 1
                    # Update sheet state to program switch to correct setting
                    if sw_state:
                        bit_loc = 11 - (y_col + 4*y_row)
                        sheet_state_h += 2**bit_loc
            sheet_states_h += [sheet_state_h]
        return sheet_states_e, sheet_states_h


# ========================= STATIC CLASS HELPER METHODS ==============================
    # calculate byte of packet metadata
    @staticmethod
    def getPacketMetadata(num_addresses,program_phase=False,program_power=False,program_mg=False):
        metadata = num_addresses
        if program_phase:
            metadata += 128
        if program_power:
            metadata += 64
        if program_mg:
            metadata += 32
        return metadata

    # calculate byte of command
    @staticmethod
    def getCommand(mg_on=False, mg_off=False, rf_on=False, rf_off=False, program_mg=False,program_phase=False,program_power=False):
        command = 0
        if mg_on:
            command += 128
        if mg_off:
            command += 64
        if rf_on:
            command += 32
        if rf_off:
            command += 16
        if program_mg:
            command += 8
        if program_phase:
            command += 4
        if program_power:
            command += 2
        return command

    # get meta gap state for a single sheet from an array of bools
    @staticmethod
    def getMetagapState(x_switch_array,y_switch_array):
        state = 0
        for x_row in range(4):
            for x_col in range(3):
                sw_state = x_switch_array[x_row][x_col]
                if sw_state:
                    bit_loc = 23 - (x_col + 3*x_row)
                    state += 2**bit_loc
        for y_row in range(3):
            for y_col in range(4):
                sw_state = y_switch_array[y_row][y_col]
                if sw_state:
                    bit_loc = 11 - (y_col + 4*y_row)
                    state += 2**bit_loc
        return state

    # covert a meta gap state for a single sheet into bytes to transfer
    @staticmethod
    def convertMetagapStateToBytes(state):
        byte1 = (state >> 16) & 255
        byte2 = (state >> 8)  & 255
        byte3 = state & 255
        return [byte1,byte2,byte3]

    # covert a phase into bytes to transfer
    @staticmethod
    def convertPhaseToBytes(phase_deg):
        while(phase_deg >= 360):
            phase_deg = phase_deg - 360
        phase = int(phase_deg*100)
        phase_msb = phase >> 8
        phase_lsb = phase-(phase_msb<<8)
        return [phase_msb,phase_lsb]
